# Test connection to FilePath REST API
# Use this to verify API credentials before installing the scheduled task

param(
    [string]$ConfigPath = "$PSScriptRoot\config.json"
)

$ErrorActionPreference = "Stop"

Write-Host ""
Write-Host "FilePath Connection Test" -ForegroundColor Cyan
Write-Host "========================" -ForegroundColor Cyan
Write-Host ""

# Load configuration
if (-not (Test-Path $ConfigPath)) {
    Write-Host "ERROR: Configuration file not found: $ConfigPath" -ForegroundColor Red
    exit 1
}

try {
    $config = Get-Content $ConfigPath -Raw | ConvertFrom-Json
}
catch {
    Write-Host "ERROR: Failed to parse config.json: $_" -ForegroundColor Red
    exit 1
}

# Validate config
if (-not $config.apiUrl) {
    Write-Host "ERROR: Missing 'apiUrl' in config.json" -ForegroundColor Red
    exit 1
}

if (-not $config.apiKey) {
    Write-Host "ERROR: Missing 'apiKey' in config.json" -ForegroundColor Red
    exit 1
}

Write-Host "API URL: $($config.apiUrl)" -ForegroundColor Gray
Write-Host "API Key: $($config.apiKey.Substring(0, [Math]::Min(8, $config.apiKey.Length)))..." -ForegroundColor Gray
Write-Host ""

# Test connection by calling GET /zones
Write-Host "Testing connection..." -ForegroundColor Yellow

$headers = @{
    "Authorization" = "Bearer $($config.apiKey)"
    "Content-Type" = "application/json"
}

try {
    $response = Invoke-RestMethod `
        -Uri "$($config.apiUrl)/zones" `
        -Method GET `
        -Headers $headers `
        -TimeoutSec 30

    Write-Host ""
    Write-Host "SUCCESS: Connection successful!" -ForegroundColor Green
    Write-Host ""

    # Display zones
    if ($response.zones -and $response.zones.Count -gt 0) {
        Write-Host "Found $($response.zones.Count) zone(s):" -ForegroundColor Cyan
        Write-Host ""

        foreach ($zone in $response.zones) {
            Write-Host "  Queue: $($zone.queueName)" -ForegroundColor White
            Write-Host "  Access Key: $($zone.accessKey)" -ForegroundColor Gray
            Write-Host "  Created: $($zone.createdAt)" -ForegroundColor Gray
            Write-Host ""
        }

        # Check which zones are configured locally
        $configuredQueues = $config.zones | Where-Object { $_.enabled } | Select-Object -ExpandProperty queueName
        $apiQueues = $response.zones | Select-Object -ExpandProperty queueName

        Write-Host "Configuration check:" -ForegroundColor Cyan
        Write-Host ""

        foreach ($queue in $configuredQueues) {
            if ($apiQueues -contains $queue) {
                Write-Host "  $queue - OK" -ForegroundColor Green
            }
            else {
                Write-Host "  $queue - WARNING: Not found in API" -ForegroundColor Yellow
            }
        }
    }
    else {
        Write-Host "No zones found in API" -ForegroundColor Yellow
        Write-Host "You may need to create zones first via the admin interface" -ForegroundColor Gray
    }

    Write-Host ""
    Write-Host "You can now install the scheduled task:" -ForegroundColor Cyan
    Write-Host "  .\install-task.ps1" -ForegroundColor White
    Write-Host ""

    exit 0
}
catch {
    Write-Host ""
    Write-Host "ERROR: Connection failed" -ForegroundColor Red
    Write-Host ""

    $statusCode = $_.Exception.Response.StatusCode.value__
    $errorMessage = $_.Exception.Message

    # Try to get detailed error from response
    if ($_.ErrorDetails.Message) {
        try {
            $errorJson = $_.ErrorDetails.Message | ConvertFrom-Json
            if ($errorJson.error) {
                $errorMessage = $errorJson.error
            }
            elseif ($errorJson.message) {
                $errorMessage = $errorJson.message
            }
        }
        catch {
            # Use default error message
        }
    }

    Write-Host "Status Code: $statusCode" -ForegroundColor Gray
    Write-Host "Error: $errorMessage" -ForegroundColor Gray
    Write-Host ""

    # Provide helpful guidance
    if ($statusCode -eq 401 -or $statusCode -eq 403 -or $errorMessage -like "*Unauthorized*" -or $errorMessage -like "*Invalid*") {
        Write-Host "HINT: Check your API key in config.json" -ForegroundColor Yellow
        Write-Host "      The API key should be your Secret Key, not Access Key" -ForegroundColor Yellow
    }
    elseif ($statusCode -eq 404) {
        Write-Host "HINT: Check the API URL in config.json" -ForegroundColor Yellow
        Write-Host "      Current: $($config.apiUrl)" -ForegroundColor Yellow
    }
    elseif ($errorMessage -like "*Could not resolve*" -or $errorMessage -like "*Unable to connect*") {
        Write-Host "HINT: Check network connectivity and DNS resolution" -ForegroundColor Yellow
    }

    Write-Host ""
    exit 1
}
