# Install FilePath Sync as a Windows Scheduled Task
# Runs every 5 minutes by default

param(
    [int]$IntervalMinutes = 5,
    [string]$TaskName = "FilePath Sync"
)

$ErrorActionPreference = "Stop"

Write-Host "Installing FilePath Sync Scheduled Task..." -ForegroundColor Cyan

# Get the path to the sync script
$scriptPath = Join-Path $PSScriptRoot "filepath-sync.ps1"
$configPath = Join-Path $PSScriptRoot "config.json"

# Verify files exist
if (-not (Test-Path $scriptPath)) {
    Write-Host "ERROR: filepath-sync.ps1 not found at: $scriptPath" -ForegroundColor Red
    exit 1
}

if (-not (Test-Path $configPath)) {
    Write-Host "ERROR: config.json not found at: $configPath" -ForegroundColor Red
    Write-Host "Please create config.json with your API credentials and zone configuration" -ForegroundColor Yellow
    exit 1
}

# Check if task already exists
$existingTask = Get-ScheduledTask -TaskName $TaskName -ErrorAction SilentlyContinue

if ($existingTask) {
    Write-Host "Task '$TaskName' already exists. Removing old task..." -ForegroundColor Yellow
    Unregister-ScheduledTask -TaskName $TaskName -Confirm:$false
}

# Create the scheduled task action
$action = New-ScheduledTaskAction `
    -Execute "powershell.exe" `
    -Argument "-NoProfile -WindowStyle Hidden -ExecutionPolicy Bypass -File `"$scriptPath`""

# Create the trigger (every X minutes, indefinitely)
$trigger = New-ScheduledTaskTrigger `
    -Once `
    -At (Get-Date) `
    -RepetitionInterval (New-TimeSpan -Minutes $IntervalMinutes) `
    -RepetitionDuration ([TimeSpan]::MaxValue)

# Create settings
$settings = New-ScheduledTaskSettingsSet `
    -AllowStartIfOnBatteries `
    -DontStopIfGoingOnBatteries `
    -StartWhenAvailable `
    -RunOnlyIfNetworkAvailable `
    -ExecutionTimeLimit (New-TimeSpan -Minutes 30) `
    -RestartCount 3 `
    -RestartInterval (New-TimeSpan -Minutes 1)

# Get current user principal
$principal = New-ScheduledTaskPrincipal `
    -UserId $env:USERNAME `
    -LogonType Interactive `
    -RunLevel Limited

# Register the task
try {
    Register-ScheduledTask `
        -TaskName $TaskName `
        -Action $action `
        -Trigger $trigger `
        -Settings $settings `
        -Principal $principal `
        -Description "FilePath sync client - synchronizes files with FilePath REST API every $IntervalMinutes minutes" | Out-Null

    Write-Host ""
    Write-Host "SUCCESS: Scheduled task installed successfully!" -ForegroundColor Green
    Write-Host ""
    Write-Host "Task Name: $TaskName" -ForegroundColor Cyan
    Write-Host "Frequency: Every $IntervalMinutes minutes" -ForegroundColor Cyan
    Write-Host "Script: $scriptPath" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "To view the task:" -ForegroundColor Yellow
    Write-Host "  Get-ScheduledTask -TaskName '$TaskName'" -ForegroundColor Gray
    Write-Host ""
    Write-Host "To test manually:" -ForegroundColor Yellow
    Write-Host "  .\filepath-sync.ps1" -ForegroundColor Gray
    Write-Host ""
    Write-Host "To uninstall:" -ForegroundColor Yellow
    Write-Host "  .\uninstall-task.ps1" -ForegroundColor Gray
    Write-Host ""
}
catch {
    Write-Host "ERROR: Failed to register scheduled task: $_" -ForegroundColor Red
    exit 1
}
