# FilePath PowerShell Client
# Synchronizes files between local directories and FilePath REST API
# Designed for Windows Server deployments

param(
    [string]$ConfigPath = "$PSScriptRoot\config.json"
)

# Configuration
$ErrorActionPreference = "Stop"
$ProgressPreference = "SilentlyContinue"

# Load configuration
function Load-Config {
    param([string]$Path)

    if (-not (Test-Path $Path)) {
        throw "Configuration file not found: $Path"
    }

    try {
        $config = Get-Content $Path -Raw | ConvertFrom-Json

        # Validate required fields
        if (-not $config.apiUrl) { throw "Missing apiUrl in config" }
        if (-not $config.apiKey) { throw "Missing apiKey in config" }
        if (-not $config.zones) { throw "Missing zones in config" }

        return $config
    }
    catch {
        throw "Failed to load config: $_"
    }
}

# Logging functions
function Write-Log {
    param(
        [string]$Message,
        [ValidateSet('INFO', 'WARN', 'ERROR')]
        [string]$Level = 'INFO'
    )

    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logMessage = "$timestamp [$Level] $Message"

    # Write to console
    switch ($Level) {
        'ERROR' { Write-Host $logMessage -ForegroundColor Red }
        'WARN'  { Write-Host $logMessage -ForegroundColor Yellow }
        default { Write-Host $logMessage }
    }

    # Write to log file if configured
    if ($script:config.logPath) {
        try {
            # Ensure log directory exists
            $logDir = Split-Path $script:config.logPath -Parent
            if ($logDir -and -not (Test-Path $logDir)) {
                New-Item -ItemType Directory -Path $logDir -Force | Out-Null
            }

            Add-Content -Path $script:config.logPath -Value $logMessage -ErrorAction SilentlyContinue
        }
        catch {
            # Silently fail if logging fails
        }
    }
}

# API Request function
function Invoke-APIRequest {
    param(
        [string]$Method,
        [string]$Path,
        [object]$Body = $null
    )

    $apiUrl = $script:config.apiUrl
    $apiKey = $script:config.apiKey

    # Build full URL
    $url = "$apiUrl/$Path".Replace("//", "/").Replace(":/", "://")

    $headers = @{
        "Authorization" = "Bearer $apiKey"
        "Content-Type" = "application/json"
    }

    try {
        $params = @{
            Uri = $url
            Method = $Method
            Headers = $headers
            TimeoutSec = 30
        }

        if ($Body) {
            $params.Body = ($Body | ConvertTo-Json -Depth 10 -Compress)
        }

        $response = Invoke-RestMethod @params
        return $response
    }
    catch {
        $statusCode = $_.Exception.Response.StatusCode.value__
        $errorMessage = $_.Exception.Message

        # Check for specific error responses
        if ($_.ErrorDetails.Message) {
            try {
                $errorJson = $_.ErrorDetails.Message | ConvertFrom-Json
                if ($errorJson.error) {
                    $errorMessage = $errorJson.error
                }
                elseif ($errorJson.message) {
                    $errorMessage = $errorJson.message
                }
            }
            catch {
                # Use default error message
            }
        }

        throw "API request failed ($statusCode): $errorMessage"
    }
}

# Download files from a zone's queue
function Get-Downloads {
    param([object]$Zone)

    if (-not $Zone.enabled) {
        return
    }

    if (-not $Zone.queueName) {
        Write-Log "Zone missing queueName, skipping" -Level WARN
        return
    }

    try {
        $response = Invoke-APIRequest -Method GET -Path "download?queueName=$([uri]::EscapeDataString($Zone.queueName))"

        if ($response -and $response.fileName) {
            # File available, download it
            $fileName = $response.fileName
            $fileContent = $response.fileContent

            # Ensure download directory exists
            $downloadPath = $Zone.downloadPath
            if (-not (Test-Path $downloadPath)) {
                New-Item -ItemType Directory -Path $downloadPath -Force | Out-Null
            }

            # Decode base64 content and save file
            $filePath = Join-Path $downloadPath $fileName
            $bytes = [System.Convert]::FromBase64String($fileContent)
            [System.IO.File]::WriteAllBytes($filePath, $bytes)

            $sizeKB = [math]::Round($bytes.Length / 1024, 2)
            Write-Log "[$($Zone.queueName)] Downloaded: $fileName ($sizeKB KB)"
        }
    }
    catch {
        $errorMsg = $_.Exception.Message

        # Don't log "No files available" as error
        if ($errorMsg -notlike "*404*" -and $errorMsg -notlike "*No files available*") {
            Write-Log "[$($Zone.queueName)] Download error: $errorMsg" -Level ERROR
        }
    }
}

# Upload files from a zone's upload directory
function Send-Uploads {
    param([object]$Zone)

    if (-not $Zone.enabled) {
        return
    }

    if (-not $Zone.uploadPath) {
        return
    }

    if (-not (Test-Path $Zone.uploadPath)) {
        return
    }

    # Load state to track uploaded files
    $stateFile = Join-Path $PSScriptRoot "upload-state.json"
    $state = @{ uploaded = @{} }

    if (Test-Path $stateFile) {
        try {
            $state = Get-Content $stateFile -Raw | ConvertFrom-Json
            if (-not $state.uploaded) {
                $state.uploaded = @{}
            }
        }
        catch {
            # If state file is corrupted, start fresh
            $state = @{ uploaded = @{} }
        }
    }

    try {
        # Get all files in upload directory
        $files = Get-ChildItem -Path $Zone.uploadPath -File -ErrorAction Stop

        foreach ($file in $files) {
            # Skip hidden files
            if ($file.Attributes -band [System.IO.FileAttributes]::Hidden) {
                continue
            }

            # Skip system files
            $systemFiles = @('desktop.ini', 'thumbs.db', '$recycle.bin')
            if ($systemFiles -contains $file.Name.ToLower()) {
                continue
            }

            # Check file size (50MB max)
            $maxSize = 50 * 1024 * 1024
            if ($file.Length -gt $maxSize) {
                $sizeMB = [math]::Round($file.Length / 1024 / 1024, 2)
                $maxMB = [math]::Round($maxSize / 1024 / 1024, 2)
                Write-Log "[$($Zone.queueName)] Skipped $($file.Name): File too large (${sizeMB}MB, max ${maxMB}MB)" -Level WARN
                continue
            }

            # Check if file is empty
            if ($file.Length -eq 0) {
                Write-Log "[$($Zone.queueName)] Skipped $($file.Name): File is empty" -Level WARN
                continue
            }

            # Calculate file hash for duplicate detection
            $fileHash = (Get-FileHash -Path $file.FullName -Algorithm SHA256).Hash
            $stateKey = "$($Zone.queueName)_$($file.Name)"

            # Check if already uploaded recently (within 5 minutes)
            if ($state.uploaded.$stateKey) {
                $lastUpload = [DateTime]$state.uploaded.$stateKey.timestamp
                $timeSince = (Get-Date) - $lastUpload

                if ($timeSince.TotalMinutes -lt 5 -and $state.uploaded.$stateKey.hash -eq $fileHash) {
                    Write-Log "[$($Zone.queueName)] Skipped $($file.Name): Already uploaded recently" -Level WARN
                    continue
                }
            }

            # Upload file
            try {
                # Read file and convert to base64
                $bytes = [System.IO.File]::ReadAllBytes($file.FullName)
                $base64 = [System.Convert]::ToBase64String($bytes)

                # Call upload API
                $body = @{
                    queueName = $Zone.queueName
                    fileName = $file.Name
                    fileContent = $base64
                }

                $response = Invoke-APIRequest -Method POST -Path "upload" -Body $body

                $sizeKB = [math]::Round($bytes.Length / 1024, 2)
                Write-Log "[$($Zone.queueName)] Uploaded: $($file.Name) ($sizeKB KB)"

                # Update state
                $state.uploaded.$stateKey = @{
                    hash = $fileHash
                    timestamp = (Get-Date).ToString("o")
                }

                # Save state
                $state | ConvertTo-Json -Depth 10 | Set-Content $stateFile

                # Delete local file after successful upload
                Remove-Item -Path $file.FullName -Force
            }
            catch {
                Write-Log "[$($Zone.queueName)] Upload failed for $($file.Name): $_" -Level ERROR
            }

            # Process one file per zone per run
            break
        }

        # Clean up old state entries (older than 10 minutes)
        $cutoff = (Get-Date).AddMinutes(-10)
        $keysToRemove = @()

        foreach ($key in $state.uploaded.Keys) {
            $timestamp = [DateTime]$state.uploaded.$key.timestamp
            if ($timestamp -lt $cutoff) {
                $keysToRemove += $key
            }
        }

        foreach ($key in $keysToRemove) {
            $state.uploaded.Remove($key)
        }

        if ($keysToRemove.Count -gt 0) {
            $state | ConvertTo-Json -Depth 10 | Set-Content $stateFile
        }
    }
    catch {
        Write-Log "[$($Zone.queueName)] Error checking upload folder: $_" -Level ERROR
    }
}

# Main execution
try {
    # Load configuration
    $script:config = Load-Config -Path $ConfigPath

    Write-Log "FilePath sync started"
    Write-Log "API: $($script:config.apiUrl)"

    # Get enabled zones
    $enabledZones = $script:config.zones | Where-Object { $_.enabled -eq $true -and $_.queueName }

    if ($enabledZones.Count -eq 0) {
        Write-Log "No enabled zones configured" -Level WARN
        exit 0
    }

    Write-Log "Processing $($enabledZones.Count) zone(s)"

    # Process each zone
    foreach ($zone in $enabledZones) {
        # Download files
        Get-Downloads -Zone $zone

        # Upload files
        Send-Uploads -Zone $zone
    }

    Write-Log "FilePath sync completed"
    exit 0
}
catch {
    Write-Log "Fatal error: $_" -Level ERROR
    exit 1
}
