# FilePath PowerShell Client

A lightweight, zero-dependency PowerShell client for synchronizing files with the FilePath service. Designed for Windows Server deployments where files need to be transferred to/from server directories.

## Features

- **Zero Dependencies**: Uses only built-in PowerShell cmdlets (PowerShell 5.1+)
- **Scheduled Execution**: Runs as Windows Scheduled Task (every 1-5 minutes)
- **Multi-Zone Support**: Configure multiple queue/directory pairs
- **Automatic Retry**: Built-in state tracking to prevent duplicate uploads
- **Logging**: Optional file-based logging for troubleshooting
- **Simple Deployment**: Just copy files and run install script

## Requirements

- Windows 7 / Windows Server 2012 or later
- PowerShell 5.1 or later (built into Windows)
- Network access to FilePath service (https://fpr.pathagility.com or fpr-dev.pathagility.com)
- FilePath credentials (Secret Key)

## Important: PowerShell Execution Policy

The FilePath PowerShell scripts are unsigned. Before running them, you must adjust your PowerShell execution policy:

```powershell
Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser
```

This allows locally created scripts to run while still requiring remote scripts to be signed. This is a standard Windows security setting.

To check your current policy:

```powershell
Get-ExecutionPolicy
```

If scripts fail to run with "cannot be loaded because running scripts is disabled", you need to adjust the execution policy as shown above.

## Quick Start

### 1. Get Your Credentials

You need a **Secret Key** from the FilePath administrator. This is the authentication token.

### 2. Configure

Edit `config.json` with your settings:

```json
{
  "apiUrl": "https://fpr.pathagility.com",
  "apiKey": "YOUR-SECRET-KEY-HERE",
  "logPath": "C:\\FilePath\\logs\\sync.log",
  "zones": [
    {
      "enabled": true,
      "queueName": "your-queue-name",
      "downloadPath": "C:\\FilePath\\Downloads",
      "uploadPath": "C:\\FilePath\\Uploads"
    }
  ]
}
```

**Configuration Fields:**

- `apiUrl` - FilePath service endpoint (dev or prod)
- `apiKey` - Your Secret Key for authentication
- `logPath` - (Optional) Path to log file
- `zones` - Array of queue/directory configurations
  - `enabled` - Set to `true` to activate this zone
  - `queueName` - Queue name from FilePath admin
  - `downloadPath` - Local directory for incoming files
  - `uploadPath` - Local directory to watch for outgoing files

### 3. Set Execution Policy

If you haven't already, allow unsigned scripts to run:

```powershell
Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser
```

### 4. Test Connection

Before installing, verify your credentials work:

```powershell
.\test-connection.ps1
```

This will:

- Validate your config.json
- Test connectivity
- Display available zones
- Verify your queue names match zones

### 5. Install Scheduled Task

Install the task to run every 5 minutes:

```powershell
.\install-task.ps1
```

Or specify a different interval:

```powershell
# Run every 2 minutes
.\install-task.ps1 -IntervalMinutes 2
```

### 6. Verify Installation

Check that the task was created:

```powershell
Get-ScheduledTask -TaskName "FilePath Sync"
```

Test manual execution:

```powershell
.\filepath-sync.ps1
```

Check the log file (if configured):

```powershell
Get-Content C:\FilePath\logs\sync.log -Tail 20
```

## Usage

### How It Works

**Downloads:**

1. Script polls service endpoint: `GET /download?queueName=your-queue`
2. If files available, downloads to `downloadPath`
3. Files are received as base64 and written to disk

**Uploads:**

1. Script scans `uploadPath` for files
2. Reads file, converts to base64
3. Posts to service endpoint: `POST /upload`
4. On success, deletes local file
5. Tracks uploads to prevent duplicates (5-minute window)

**Scheduling:**

- Task Scheduler runs script every N minutes
- Each run processes one file per zone (to avoid timeout)
- State file tracks recently uploaded files

### File Processing Rules

**Files that are SKIPPED:**

- Hidden files (including `.` prefix on Windows)
- System files (desktop.ini, thumbs.db, etc.)
- Empty files (0 bytes)
- Files larger than 50MB
- Recently uploaded files (within 5 minutes, same hash)

**Files are processed:**

- One at a time per zone per run
- In alphabetical order
- Deleted after successful upload

### Monitoring

**Check scheduled task status:**

```powershell
Get-ScheduledTask -TaskName "FilePath Sync" | Get-ScheduledTaskInfo
```

**View recent runs:**

```powershell
Get-ScheduledTask -TaskName "FilePath Sync" | Get-ScheduledTaskInfo | Select-Object LastRunTime, LastTaskResult, NextRunTime
```

**Manually trigger task:**

```powershell
Start-ScheduledTask -TaskName "FilePath Sync"
```

**View logs:**

```powershell
# Tail log file (if configured)
Get-Content C:\FilePath\logs\sync.log -Tail 50 -Wait

# View all logs from today
Get-Content C:\FilePath\logs\sync.log | Select-String (Get-Date -Format "yyyy-MM-dd")
```

## Troubleshooting

### Execution Policy Issues

**Error: "File cannot be loaded because running scripts is disabled on this system"**

You need to adjust PowerShell's execution policy:

```powershell
Set-ExecutionPolicy -ExecutionPolicy RemoteSigned -Scope CurrentUser
```

**Error: "Execution policy change rejected"**

You need to run PowerShell as Administrator to change execution policy, or use `-Scope CurrentUser` as shown above.

**Check current policy:**

```powershell
Get-ExecutionPolicy
Get-ExecutionPolicy -List
```

**Alternative: Bypass execution policy for single run**

If you cannot change the execution policy permanently, you can bypass it for a single run:

```powershell
PowerShell.exe -ExecutionPolicy Bypass -File .\filepath-sync.ps1
```

Note: If using this method, you'll need to modify `install-task.ps1` to include the `-ExecutionPolicy Bypass` parameter in the task configuration.

### Connection Issues

**Error: "401 Unauthorized" or "403 Forbidden"**

- Check that `apiKey` in config.json is correct
- Make sure you're using the **Secret Key**, not Access Key
- Verify key hasn't been revoked

**Error: "404 Not Found"**

- Check `apiUrl` is correct
- For dev: `https://fpr-dev.pathagility.com`
- For prod: `https://fpr.pathagility.com`

**Error: "Could not resolve host"**

- Check network connectivity
- Verify DNS resolution: `Resolve-DnsName fpr.pathagility.com`
- Check firewall rules

### File Issues

**Files not uploading:**

1. Check that `uploadPath` exists and has files
2. Verify files aren't hidden or system files
3. Check file size (max 50MB)
4. Review log for error messages
5. Run script manually to see immediate output: `.\filepath-sync.ps1`

**Files not downloading:**

1. Verify `downloadPath` exists and is writable
2. Check that queue has files available (ask administrator)
3. Review log for error messages
4. Test connection: `.\test-connection.ps1`

**Duplicate uploads:**

- Script tracks uploads for 5 minutes using file hash
- State stored in `upload-state.json`
- If duplicates still occur, check if file content is actually changing

### Task Scheduler Issues

**Task not running:**

```powershell
# Check task configuration
Get-ScheduledTask -TaskName "FilePath Sync" | Format-List *

# Check last run result (0 = success, non-zero = error)
Get-ScheduledTask -TaskName "FilePath Sync" | Get-ScheduledTaskInfo

# View task history in Event Viewer
Get-WinEvent -LogName "Microsoft-Windows-TaskScheduler/Operational" | Where-Object { $_.Message -like "*FilePath*" } | Select-Object -First 10
```

**Task exists but not running:**

- Check that "Run only if network is available" setting isn't blocking execution
- Verify user account has permissions to run scheduled tasks
- Check execution policy: `Get-ExecutionPolicy`

## Uninstall

To remove the scheduled task:

```powershell
.\uninstall-task.ps1
```

This only removes the task. To completely uninstall, also delete the script directory.

## Advanced Configuration

### Multiple Zones

You can configure multiple zones in config.json:

```json
{
  "zones": [
    {
      "enabled": true,
      "queueName": "lab-results",
      "downloadPath": "D:\\LabResults\\Incoming",
      "uploadPath": "D:\\LabResults\\Outgoing"
    },
    {
      "enabled": true,
      "queueName": "imaging-files",
      "downloadPath": "D:\\Imaging\\Incoming",
      "uploadPath": "D:\\Imaging\\Outgoing"
    }
  ]
}
```

### Custom Polling Interval

Change how often the task runs:

```powershell
# Check every minute (very frequent)
.\install-task.ps1 -IntervalMinutes 1

# Check every 10 minutes (less frequent)
.\install-task.ps1 -IntervalMinutes 10
```

**Note:** Files are processed one at a time per zone. More frequent polling = faster file transfers.

### Running as Different User

To run under a service account:

1. Edit the task after installation:
   
   ```powershell
   # Open Task Scheduler
   taskschd.msc
   ```

2. Find "FilePath Sync" task

3. Right-click → Properties → General tab

4. Click "Change User or Group"

5. Enter service account credentials

6. Check "Run whether user is logged on or not"

### Multiple Config Files

Run with a different config file:

```powershell
.\filepath-sync.ps1 -ConfigPath "C:\FilePath\custom-config.json"
```

To use custom config with scheduled task, edit `install-task.ps1` and modify the `-Argument` parameter.

## Files

- `filepath-sync.ps1` - Main sync script
- `config.json` - Configuration file (credentials and zones)
- `install-task.ps1` - Installs scheduled task
- `uninstall-task.ps1` - Removes scheduled task
- `test-connection.ps1` - Tests connectivity
- `upload-state.json` - (Auto-created) Tracks uploaded files to prevent duplicates
- `README.md` - This file

## Security Notes

- Service key stored in plaintext in config.json
- Ensure config.json has appropriate file permissions
- Use Windows ACLs to restrict access: `icacls config.json /grant Administrators:F /inheritance:r`
- Consider encrypting the config.json directory
- Log files may contain file names (but not content)

## Support

For issues or questions:

1. Check troubleshooting section above
2. Review log files for error messages
3. Test connection with `test-connection.ps1`
4. Contact your FilePath administrator

## License

Proprietary - Pathagility LLC
